/*
 * Copyright (C) 2012 Edge-Core Networks
 * This software file (the "File") is owned and distributed by 
 * Edge-Core Networks under the following licensing terms.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#ifndef _AMS_PART_H_
#define _AMS_PART_H_

#ifdef CONFIG_AMS_UBOOT
#include <ams_common.h>
#endif

#define FLASH_REGION_NAMELEN        64
#define PART_FILE_NAMELEN           64
#define DEVICE_NAMELEN              64
#define MAX_PARTITION_NUMBER	    16
#define PARTITION_TABLE_MAGIC       0x646c7470
#define PARTITION_VERSION           0x1
#define PARTITION_TYPE_LEN          16
#define CF_STANDARD_SECTOR_SIZE		512
#define AMS_BPB_SIGNATURE           0x55AAAA55

#define DEVICE_NOR_NAME             "nor"
#define DEVICE_NAND_NAME            "nand"
#define DEVICE_CF_NAME              "cfcard"
#define DEVICE_SPI_NAME             "spi"   /* 2009/07/22, aaron, add new flash device (SPI flash) in AMS */

/* partition mode */
#define PART_MODE_RO    0x00        /* Read Only */
#define PART_MODE_RW    0x01        /* Read + Write */


/* 2009/08, aaron, force size of enum to 4 bytes
 * because not all of compiler can take enum to be the same size.
 */
typedef unsigned long AMS_PARTITION_TYPE_T;
typedef unsigned long AMS_FILEMAPPING_PART_TYPE_T;
typedef unsigned long AMS_FILEMAPPING_PARTITION_STATUS_T;
typedef unsigned long AMS_PARTITION_DEVICE_TYPE_T;


 /*
  *  Partition table structure define, also used as file type
  */
enum
{
    TYPE_NOT_USED,              /* partition not used */
    TYPE_BOOTLOADER,            /* bootloader */
    TYPE_KERNEL,                /* kernel image */
    TYPE_RUNTIME,               /* runtime Image */
    TYPE_ENV,                   /* u-boot env parames */
    TYPE_ROOT_FS,               /* root file system */
    TYPE_PARTITION_TABLE,       /* partition table management */
    TYPE_FILEMAPPING_TABLE,     /* filemapping management */
    TYPE_USERDATA,              /* user data */
    TYPE_HWINFO,                /* hardware information */
    TYPE_FS,                    /* filesystem management:ext2 */
    TYPE_BACKUP_BOOT_RECORD,    /* for backup critical data, partition table, etc. */
    TYPE_MAX                    /* type end*/
};  /* AMS_PARTITION_TYPE_T */

 /*
  *  Partition device type define
  */
enum
{
	TYPE_DEVICE_NO_USED,
	TYPE_DEVICE_NOR,
	TYPE_DEVICE_NAND,
	TYPE_DEVICE_CFCARD,
	TYPE_DEVICE_SPI,             /* 2009/07/22, aaron, add new flash device (SPI flash) in AMS */
	TYPE_DEVICE_MAX
}; /* AMS_PARTITION_DEVICE_TYPE_T */

 /*
  *  filemapping partition type define
  */
enum
{
    PART_TYPE_NO_USED,
    PART_TYPE_FIXED,
    PART_TYPE_FILESYSTEM
}; /* AMS_FILEMAPPING_PART_TYPE_T */

 /*
  *  filemapping partition status define
  */
enum
{
    AMS_PARTITION_FREE = 1,
    AMS_PARTITION_ACTIVE = 2,
    AMS_PARTITION_INACTIVE = 4,
}; /* AMS_FILEMAPPING_PARTITION_STATUS_T */


 /*
  *  Partition Table Management struct define
  */

typedef struct FlashRegion {
    unsigned char  region_name[FLASH_REGION_NAMELEN];
    unsigned long  region_base;
    unsigned long  region_size;
    unsigned long  region_type;                 /* indicate partition type */
    unsigned long  region_mode;                 /* region mode, R/W */                
    unsigned long  region_reserved[7];          /* reserved 28 bytes */     
    unsigned char  device_name[DEVICE_NAMELEN];
    unsigned long  region_checksum;
} FlashRegion_T;

/*
 *  Flash Partition Table type
 */
typedef struct PartitionTable {
    unsigned long magic; 	          /* BOOTLDR_PARTITION_MAGIC should be filled with 0x646c7470 (btlp) */
    unsigned long version;			  /* version number   */
    unsigned long reserved[2];		  /* reserved 8 bytes */
    unsigned long header_checksum;    /* checksum of header in this data structure */
    struct FlashRegion flash_region[MAX_PARTITION_NUMBER];
} AMS_PARTITION_TABLE_T;



/*
 *  FileMapping Management type
 */
typedef struct FileMapping{                  
    unsigned char file_name[PART_FILE_NAMELEN]; /* filename in partition */
    unsigned long file_size;                    /* file size */
    unsigned long creat_time;                   /* file create time */
    unsigned long part_index;                   /* file-saved partition index */
    AMS_PARTITION_TYPE_T               file_type;       /* file type */
    AMS_FILEMAPPING_PART_TYPE_T        part_type;       /* file-saved partition type */
    AMS_FILEMAPPING_PARTITION_STATUS_T startup_flag;    /* start-up flag  indicate partition status is active or free */
    unsigned long reserved[40];                         /* reserved 160 bytes for runtime use */
    unsigned long part_checksum;                /* checksum of this data structure */
}AMS_FILEMAPPING_T;


typedef struct FileMappingManage{
    struct FileMapping filemapping[MAX_PARTITION_NUMBER];
}AMS_FILEMAPPING_MGR_T;

typedef struct PartBootConfig {
    unsigned long  bootflags;
    unsigned long  reserve[64];
} AMS_PARTBOOTCONFIG_T;


/*
 * Boot Parameter Block type
 */
typedef struct BPB_S{
    unsigned long           signature;      /* Signature */
    unsigned long           length;         /* byte count of field except for checksum */
    AMS_PARTITION_TABLE_T   pat_table;      /* partition table */
    AMS_FILEMAPPING_MGR_T   file_map;       /* file mapping table */
    FS_HW_Info_T            hw_info;        /* hardware information */
    unsigned long           checksum;           
}AMS_BPB_T;

/* functon description */
unsigned long AMS_PART_get_device_type(const char *device_name, AMS_PARTITION_DEVICE_TYPE_T *device_type);
unsigned long AMS_PART_get_filemap_elememt(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr);
unsigned long AMS_PART_get_filemapping_region(FlashRegion_T *region_ptr);
unsigned long AMS_PART_init_filemapping(unsigned long index, AMS_FILEMAPPING_T *filemapping_ptr);
unsigned long AMS_PART_get_partition_table_ptr(AMS_PARTITION_TABLE_T **ptr);
unsigned long AMS_PART_get_filemapping_mgr_ptr(AMS_FILEMAPPING_MGR_T **ptr);
int AMS_PART_get_default_filemapp_table(AMS_FILEMAPPING_MGR_T *filemap);
int AMS_PART_get_default_part_table(AMS_PARTITION_TABLE_T *part_table);
unsigned long AMS_PART_check_active(unsigned long index, AMS_PARTITION_TYPE_T type);
unsigned long AMS_PART_get_startup_by_type(AMS_PARTITION_TYPE_T type,  unsigned long *partiton_idx);
unsigned long AMS_PART_get_info_by_index(unsigned long index, unsigned long *addr, unsigned long *size);
unsigned long AMS_PART_get_info_by_type(AMS_PARTITION_TYPE_T type, unsigned long *addr, unsigned long *size, unsigned long part_index, AMS_PARTITION_DEVICE_TYPE_T *device_type);
unsigned long AMS_PART_set_part_info_by_index(unsigned long part_idx, const char *name, unsigned long file_size, AMS_PARTITION_TYPE_T type);
unsigned long AMS_PART_get_info_by_name(unsigned char *name, unsigned long *addr, unsigned long *size, AMS_PARTITION_TYPE_T *type);
unsigned long AMS_PART_get_info_by_filename(const char *filename, unsigned long *addr, unsigned long *size, unsigned long *part_idx, AMS_PARTITION_DEVICE_TYPE_T *device_type);
unsigned long AMS_PART_get_partition_filename(unsigned long index, unsigned char *file_name, unsigned long *file_length);
unsigned long AMS_PART_write_part2flash(AMS_PARTITION_TABLE_T *partition_table_ptr);
unsigned long AMS_PART_write_filemapping2flash(AMS_FILEMAPPING_MGR_T *filemapping_mgr);
unsigned long AMS_PART_get_region(unsigned long index, FlashRegion_T *region);
unsigned long AMS_PART_check_partition(void);
unsigned long AMS_PART_write_to_device(AMS_PARTITION_TYPE_T type, const char *name, unsigned long addr, unsigned long length, unsigned long part_index);
unsigned long AMS_PART_read_from_device(AMS_PARTITION_TYPE_T type, const char *name, unsigned long length, unsigned long *buf_addr);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_non_startup_filename
 * PURPOSE  : get the inactive file of a given type.
 * INPUT    : file_type   -- the type of file
 * OUTPUT   : 
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   : 
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_non_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_get_startup_filename
 * PURPOSE  : get the startup file of a given type.
 * INPUT    : 
 *            file_type   -- the type of file
 * OUTPUT   :
 *            file_name   -- the file name 
 *            file_length -- the size of file.
 * RETURN   :
 *            RC_OK    = Success
 *            RC_ERROR = Fail
 * NOTES    : None
 * ------------------------------------------------------------------------
 */
int AMS_PART_get_startup_filename(AMS_PARTITION_TYPE_T file_type, unsigned char *file_name, unsigned long *file_length);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_recover_runtime_to_default
 * PURPOSE  :   Recover runtime file information and
 *              write the default file name, size.
 *              Bootable runtime image will be loaded at LA_DB_ADDR if it is
 *              recovered successfully.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    :   This function shall check the runtime file in flash first.
 *              If runtime's checksum is correct, then using default file name.
 *              If there is environment "runtimename", get file name from "runtimename".
 *              Otherwise, runtime file name is the partition name with extension ".bix".
 *
 *              If runtime's checksum is not correct, then check next runtime file.
 * ------------------------------------------------------------------------
 */
int AMS_PART_recover_runtime_to_default(void);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_validate_and_load_boot_runtime
 * PURPOSE  :   Validate the startup runtime image(the runtime image set as active)
 *              The runtime image will be loaded to LA_DB_ADDR after returning
 *              from this function if there is no error.
 * INPUT    :   None
 * OUTPUT   :   None
 * RETURN   :   
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 *
 * NOTES    : 1.This function will validate the startup runtime image through
 *              image header checksum and image data checksum. The runtime
 *              image will be loaded to LA_DB_ADDR if there is no error.
 *            2.Invalid runtime image will be deleted from file mapping table.
 * ------------------------------------------------------------------------
 */
int AMS_PART_validate_and_load_boot_runtime(void);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_delete_runtime_file
 * PURPOSE:	delete runtime file in flash device.
 * INPUT:	file name
 * OUTPUT:	None
 * RETURN:	
 *              RC_OK    = Success
 *              RC_ERROR = Fail
 * NOTES:
 * ------------------------------------------------------------------------
 */
int AMS_PART_delete_runtime_file(char *file_name);

/* ------------------------------------------------------------------------
 * FUNCTION NAME: AMS_PART_prepare_runtime_image
 * PURPOSE: Check file partition table and select a bootable runtime image.
 *          If a bootable runtime image is not available, this function will
 *          try to recover a usable runtime image in flash through the
 *          validation of checksum.
 * INPUT:   None
 * OUTPUT:  boot_runtime_filename -- the filename of the runtime that is
 *                                   being used to boot up.
 *                                   The size of boot_runtime_filename must be
 *                                   greater or equal to PART_FILE_NAMELEN.
 * RETURN:  RC_OK    = Success
 *          RC_ERROR = Fail (No runtime image availabe for boot)
 * NOTES: 1.If RC_OK is returned, the bootable runtime image will be loaded
 *          at LA_DB_ADDR.
 *        2.Invalid runtime will be deleted from file mapping table.
 *        3.The startup runtime could be changed if the original startup
 *          runtime image is invalid.
 * ------------------------------------------------------------------------
 */
int AMS_PART_prepare_runtime_image(char* boot_runtime_filename);

unsigned long AMS_PART_get_boot_part( AMS_PARTITION_TYPE_T type,unsigned long *active_part_idx);
//unsigned long AMS_PART_get_region_mode(FlashRegion_T* region);
int AMS_PART_find_filemap_element_checksum(AMS_FILEMAPPING_T *filemap, unsigned long *chksum);
int AMS_PART_init(AMS_PARTITION_TABLE_T *def_pt, FS_HW_Info_T *def_hit);
int AMS_PART_get_hwinfo(FS_HW_Info_T *hit);
int AMS_PART_get_inactive_part(AMS_PARTITION_TYPE_T type, unsigned long *part_idx);
int AMS_PART_write_hwinfo(FS_HW_Info_T *hit);
int AMS_PART_reset(void);
int AMS_PART_reset_filemap(void);
int AMS_PART_reset_hwinfo(void);




/* software flash chip selection */
#ifdef CONFIG_AMS_FLASH_CS
extern ulong OEM_flash_v2p(ulong address);
#define AMS_FLASH_MAPPING(vaddr, addr)     { addr = OEM_flash_v2p(vaddr); }
#define AMS_FLASH_DEFAULT()         OEM_flash_v2p(CFG_FLASH_BASE)
#else
#define AMS_FLASH_MAPPING(vaddr, addr)     (addr=vaddr)
#define AMS_FLASH_DEFAULT()
#endif

#endif  /* #_AMS_PART_H_ */

